% This m-file is for calculating the 2D on detector psf for a single dipole
% emitter using the czt-function with arbitrary windows and discretization
% in direct and Fourier space.

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands   

function [allPSFs, Duxstr, Duystr, Duzstr] = ...
    PSFgenerator(settings, aberr, fixed, singleorientation, Ncfg)

% parameters: NA, refractive indices of medium, cover slip, immersion fluid,
% wavelength (in nm), magnification, pixel size (Nyquist = 1/4).
if nargin<1
   model = 'standard2D';
   NA = 1.25;
   refmed = 1.33;
   refcov = 1.50;
   refimm = 1.50;
   lambda = 500.0;
   pitch = 1/2;
   phasedepth = pi/2;
   pixelsize = 0.25;
   Nxpixels = 11;
   Nypixels = 11;
   Kpoints = 3;
   Nxpupil = 123;
   Nypupil = 123;
   pola = 90.0;
   azim = 45.0;
else
   model = settings.model;
   NA = settings.NA;
   refmed = settings.refmed;
   refcov = settings.refcov;
   refimm = settings.refimm;
   lambda = settings.lambda;
   pitch = settings.pitch;
   phasedepth = settings.phasedepth;
   scurvelength = settings.scurvelength;
   dof = settings.dof;
   pixelsize = settings.pixelsize;
   Nxpixels = settings.Nxpixels;
   Nypixels = settings.Nypixels;
   Kpoints = settings.Kpoints;
   Nxpupil = settings.Nxpupil;
   Nypupil = settings.Nypupil;
   pola = settings.pola;
   azim = settings.azim;
end
if nargin<3; fixed = 1;end
if nargin<4; Ncfg = 1000;end
if nargin<2
 defocus = 0.0;
 diagast = 0.0;
 horverast = 0;
 xcoma = 0.00;
 ycoma = 0.00;
 sphab = 0.0;
 scurvelength = 0;
else
 defocus = aberr.defocus;
 diagast = aberr.diagast;
 horverast = aberr.horverast;
 xcoma = aberr.xcoma;
 ycoma = aberr.ycoma;
 sphab = aberr.sphab;
 scurvelength = aberr.scurvelength;
end

% number of points per pixel, total number of points in image plane and
% pupil plane.
Mximage = Nxpixels*Kpoints;
Myimage = Nypixels*Kpoints;

% Transform S-curve length (= distance between astigmatic lines for
% astigmatic 3D localizations-curve length (= distance between astigmatic
% focal lines) in image space in nm.
f1av = 1/3;
f1sqav = 1/5;
NAmed = NA/refmed;
f2av = refmed*(1-sqrt(1-NAmed^2)/2-asin(NAmed)/2/NAmed); 
f2sqav = refmed^2*(2-NAmed^2/3-sqrt(1-NAmed^2)-asin(NAmed)/NAmed);
f1f2av = refmed*(1/3-(NAmed*sqrt(1-NAmed^2)*(2*NAmed^2-1)+asin(NAmed))/8/NAmed^3);
Sfac = (f2sqav-f2av^2)/(2*sqrt(6)*(f1f2av-f1av*f2av));
horverast = horverast+Sfac*scurvelength/lambda;

% Generation of Ncfg random dipole orientations and emitter positions.
% Uniform distribution in cos(pola) and azim, so uniform distribution
% in solid angle, normal distribution for lateral emitter position with
% zero mean and standard deviation pixelsize/2=lambda/8/NA for Nyquist
% sampling. Uniform distribution for axial emitter position between the
% two focal lines.

if (singleorientation)
    cospolastr = cos(pola*pi/180)*ones(Ncfg,1);
    azimstr = (pi*azim/180)*ones(Ncfg,1);
else
    cospolastr = -1+2*rand(Ncfg,1);
    azimstr = pi*rand(Ncfg,1);
end
Duxstr = 0*pixelsize*(2*rand(Ncfg,1)-1);
Duystr = 0*pixelsize*(2*rand(Ncfg,1)-1);
switch model
  case {'standard2D','diffractivecolor2D'}
    Duzstr = zeros(Ncfg,1);
  case {'astigmatic3D','diffractivecolor3D'}
    Duzstr = (scurvelength/2/lambda)*(2*rand(Ncfg,1)-1);
%     Duzstr = -(sqrt((scurvelength/2)^2+dof^2)/lambda)*ones(Ncfg,1);
end

% pupil and image size and coordinate sampling
PupilSize = 1.0;
DxPupil = 2*PupilSize/Nxpupil;
DyPupil = 2*PupilSize/Nypupil;
xpup = -PupilSize+DxPupil/2:DxPupil:PupilSize;
ypup = -PupilSize+DyPupil/2:DyPupil:PupilSize;
[XPupil,YPupil] = meshgrid(xpup,ypup);
ImageSizex = Nxpixels*pixelsize/2;
ImageSizey = Nypixels*pixelsize/2;
DxImage = 2*ImageSizex/Mximage;
ximag = -ImageSizex+DxImage/2:DxImage:ImageSizex;
DyImage = 2*ImageSizey/Myimage;
yimag = -ImageSizey+DyImage/2:DyImage:ImageSizey;
[XImage,YImage] = meshgrid(ximag,yimag);

% constants and array needed for chirpz from pupil to image
wix=exp(1i*8*pi*PupilSize*ImageSizex/Nxpupil/Mximage);
wiy=exp(1i*8*pi*PupilSize*ImageSizey/Nypupil/Myimage);
aix=exp(1i*4*pi*PupilSize*ImageSizex*(Mximage-1)/Nxpupil/Mximage);
aiy=exp(1i*4*pi*PupilSize*ImageSizey*(Myimage-1)/Nypupil/Myimage);
cix=(2/sqrt(pi))*exp(1i*2*pi*PupilSize*ImageSizex*(Nxpupil-1)*(Mximage-1)/Nxpupil/Mximage);
ciy=(2/sqrt(pi))*exp(1i*2*pi*PupilSize*ImageSizey*(Nypupil-1)*(Myimage-1)/Nypupil/Myimage);
[YKcori,~] = meshgrid(0:Myimage-1,0:Nxpupil-1);
[~,XKcori] = meshgrid(0:Myimage-1,0:Mximage-1);
CorrectionImageX=exp(-1i*4*pi*(PupilSize*ImageSizex*(Nxpupil-1)/Nxpupil/Mximage)*XKcori)*cix*PupilSize/Nxpupil;
CorrectionImageY=exp(-1i*4*pi*(PupilSize*ImageSizey*(Nypupil-1)/Nypupil/Myimage)*YKcori)*ciy*PupilSize/Nypupil;

% definition aperture
NAratio = 1.0;
ApertureMask = double((XPupil.^2+YPupil.^2)<NAratio^2);

% calculation of relevant Fresnel-coefficients for the interfaces
% between the medium and the cover slip and between the cover slip
% and the immersion fluid.
CosThetaMed = sqrt(1-(XPupil.^2+YPupil.^2)*NA^2/refmed^2);
CosThetaCov = sqrt(1-(XPupil.^2+YPupil.^2)*NA^2/refcov^2);
CosThetaImm = sqrt(1-(XPupil.^2+YPupil.^2)*NA^2/refimm^2);
pmed = refmed./CosThetaMed;
smed = refmed.*CosThetaMed;
pcov = refcov./CosThetaCov;
scov = refcov.*CosThetaCov;
pimm = refimm./CosThetaImm;
simm = refimm.*CosThetaImm;
prfcmedcov = (refmed+refcov)/refmed;
FresnelPmedcov = prfcmedcov*pmed./(pmed+pcov);
FresnelSmedcov = prfcmedcov*smed./(smed+scov);
prfccovimm = (refcov+refimm)/refcov;
FresnelPcovimm = prfccovimm*pcov./(pcov+pimm);
FresnelScovimm = prfccovimm*scov./(scov+simm);
FresnelP = FresnelPmedcov.*FresnelPcovimm;
FresnelS = FresnelSmedcov.*FresnelScovimm;

% setting of vectorial functions
Phi = atan2(YPupil,XPupil);
CosPhi = cos(Phi);
SinPhi = sin(Phi);
CosTheta = sqrt(1-(XPupil.^2+YPupil.^2)*NA^2/refmed^2);
SinTheta = sqrt(1-CosTheta.^2);

pvec{1} = FresnelP.*CosTheta.*CosPhi;
pvec{2} = FresnelP.*CosTheta.*SinPhi;
pvec{3} = -FresnelP.*SinTheta;
svec{1} = -FresnelS.*SinPhi;
svec{2} = FresnelS.*CosPhi;
svec{3} = 0;

polvec = cell(2,3);
for jtel = 1:3
  polvec{1,jtel} = CosPhi.*pvec{jtel}-SinPhi.*svec{jtel};
  polvec{2,jtel} = SinPhi.*pvec{jtel}+CosPhi.*svec{jtel};
end

% initialization arrays
ImagingPSF{1,1} = zeros(size(XImage));
ImagingPSF{1,2} = zeros(size(XImage));
ImagingPSF{1,3} = zeros(size(XImage));
ImagingPSF{2,1} = zeros(size(XImage));
ImagingPSF{2,2} = zeros(size(XImage));
ImagingPSF{2,3} = zeros(size(XImage));

% setting of aberrations (defocus, astigmatism, coma, spherical aberration)
Rho2 = XPupil.^2+YPupil.^2;

defocus = defocus*sqrt(3);
Wdefocus = defocus*(2*Rho2-1);
horverast = horverast*sqrt(6);
diagast = diagast*sqrt(6);
Wastig = horverast*(XPupil.^2-YPupil.^2)+diagast*2.*XPupil.*YPupil;
xcoma = xcoma*sqrt(8);
ycoma = ycoma*sqrt(8);
Wcoma = (xcoma*XPupil+ycoma*YPupil).*(3*Rho2-2);
sphab = sphab*sqrt(5);
Wsphab = sphab*(6*Rho2.^2-6*Rho2+1);
Waberration = Wdefocus+Wastig+Wcoma+Wsphab;

allPSFs = zeros(Nypixels,Nxpixels,Ncfg);

% loop over random configurations.
for ii = 1:Ncfg
    
% randomly selected emitter dipole orientation 
cospola = cospolastr(ii);
azim = azimstr(ii);
sinpola = sqrt(1-cospola^2);
dipor(1) = sinpola*cos(azim);
dipor(2) = sinpola*sin(azim);
dipor(3) = cospola;

% randomly selected emitter position.
Dux = Duxstr(ii);
Duy = Duystr(ii);
Duz = Duzstr(ii);

% phase contribution due to position of the emitter.
Wx = -Dux*XPupil;
Wy = -Duy*YPupil;
Wz = Duz*Rho2./(1+sqrt(1-Rho2*(NA/refmed)^2));
Wpos = Wx+Wy+Wz;
TotalPhase = Wpos+Waberration;
PhaseMask = exp(2*pi*1i*TotalPhase);

% contribution due to binary grating
switch model
  case 'diffractivecolor2D'
    ZoneFunction = XPupil/pitch;
    GratingMask = exp(pi*1i*phasedepth*sign(cos(2*pi*ZoneFunction))/lambda);
    PhaseMask = GratingMask.*PhaseMask;
  case 'diffractivecolor3D'
    offset = 0.25;  
    ZoneFunction = offset+XPupil/pitch+(scurvelength/lambda/2)*ApertureMask.*Rho2./(1+sqrt(1-Rho2*(NA/refmed)^2));
    GratingMask = exp(pi*1i*phasedepth*sign(cos(2*pi*ZoneFunction))/lambda);
    PhaseMask = GratingMask.*PhaseMask;
end

% pupil functions and image plane complex amplitude PSF
for itel = 1:2
  for jtel = 1:3
   PupilFunction = ApertureMask.*PhaseMask.*polvec{itel,jtel};
   IntermediateImage = CorrectionImageY.*transpose(czt(PupilFunction,Myimage,wiy,aiy));
   ImagingPSF{itel,jtel} = transpose(CorrectionImageX.*czt(IntermediateImage,Mximage,wix,aix));
  end
end

% calculation of dipole orientation averaged PSF
AveragePSF = zeros(Myimage,Mximage);
for itel = 1:2
    for jtel = 1:3
       AveragePSF = AveragePSF + abs(ImagingPSF{itel,jtel}).^2;
    end
end

% calculation of fixed dipole PSF
Ex = dipor(1)*ImagingPSF{1,1}+dipor(2)*ImagingPSF{1,2}+dipor(3)*ImagingPSF{1,3};
Ey = dipor(1)*ImagingPSF{2,1}+dipor(2)*ImagingPSF{2,2}+dipor(3)*ImagingPSF{2,3};
FixedPSF = abs(Ex).^2+abs(Ey).^2;
   
if (fixed == 0)
   tmpPSF = AveragePSF;
end
if (fixed == 1)
   tmpPSF = FixedPSF;  
end
if (fixed == 2)
   tmpPSF = (3*AveragePSF-FixedPSF)/2;  
end

% convolution with finite pixel size by averaging over Kpoints X Kpoints
% grid points per pixel.
convPSF = zeros(Nypixels,Nxpixels);
for iy = 1:Nypixels
  for ix = 1:Nxpixels
    convPSF(iy,ix) = sum(sum(tmpPSF((iy-1)*Kpoints+1:iy*Kpoints,(ix-1)*Kpoints+1:ix*Kpoints)));
  end
end

% normalization of the PSF
norm = sum(sum(convPSF));
allPSFs(:,:,ii) = convPSF./norm;


end
